// File: app/[your-post-type]/category/[slug]/page.tsx
// Replace [your-post-type] with your actual custom post type name

"use client";

import { use, useEffect, useState } from "react";
import { notFound } from "next/navigation";
import wordpressAPI from "@/lib/wordpress";
import ContentCard from "@/components/cards/ContentCard";
import Pagination from "@/components/common/Pagination";
import SectionHeader from "@/components/common/SectionHeader";
import { useLanguage } from "@/contexts/LanguageContext";

interface CustomPostTypeCategoryProps {
  params: Promise<{ slug: string }>;
}

// 🚨 IMPORTANT: Update these constants for your custom post type
const POST_TYPE = 'portfolio'; // Change to your custom post type name (e.g., 'services', 'portfolio', etc.)
const TAXONOMY = 'portfolios'; // Change to your custom taxonomy name
const POST_TYPE_LABEL = 'Portfolio'; // Human readable label for your post type

export default function CustomPostTypeCategoryArchive({ params }: CustomPostTypeCategoryProps) {
  const { slug } = use(params);
  const { language } = useLanguage();

  // State management
  const [posts, setPosts] = useState<any[]>([]);
  const [category, setCategory] = useState<any | null>(null);
  const [loading, setLoading] = useState(true);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);

  // Posts per page limit
  const LIMIT = parseInt(process.env.NEXT_PUBLIC_POSTS_LIMIT || "6", 10);

  // Fetch category info and posts
  useEffect(() => {
    const fetchCategoryData = async () => {
      try {
        setLoading(true);

        // Fetch taxonomy term info (try both methods)
        let termInfo = null;
        try {
          termInfo = await wordpressAPI.getCustomTaxonomyTermBySlug(TAXONOMY, slug, language);
        } catch (error) {
          console.log('Primary method failed, trying alternative...');
          termInfo = await wordpressAPI.getCustomTaxonomyTermBySlugFromPosts(POST_TYPE, TAXONOMY, slug, language);
        }

        if (!termInfo) {
          notFound();
          return;
        }
        setCategory(termInfo);

        // Fetch posts by taxonomy term (try both methods)
        let data;
        try {
          data = await wordpressAPI.getPostsByCustomTaxonomy(
            POST_TYPE,
            TAXONOMY,
            slug,
            language,
            LIMIT,
            currentPage
          );
        } catch (error) {
          console.log('Primary method failed, trying alternative...');
          data = await wordpressAPI.getPostsByCustomTaxonomyAlternative(
            POST_TYPE,
            TAXONOMY,
            slug,
            language,
            LIMIT,
            currentPage
          );
        }
        setPosts(data.items || []);
        setTotalPages(data.totalPages || 1);
      } catch (error) {
        console.error("Error loading category data:", error);
        setPosts([]);
        setTotalPages(1);
      } finally {
        setLoading(false);
      }
    };

    fetchCategoryData();
  }, [slug, language, currentPage]);

  // Loading state
  if (loading && !category) {
    return (
      <div className="te-category-archive my-12">
        <div className="container">
          <div className="animate-pulse space-y-6">
            <div className="space-y-2">
              <div className="h-8 bg-gray-200 dark:bg-gray-700 rounded w-1/3" />
              <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-2/3" />
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {Array.from({ length: LIMIT }).map((_, i) => (
                <div key={i} className="animate-pulse bg-white dark:bg-gray-800 rounded-lg shadow-md overflow-hidden">
                  <div className="aspect-[3/2] w-full bg-gray-200 dark:bg-gray-700" />
                  <div className="p-6 space-y-4">
                    <div className="h-6 bg-gray-200 dark:bg-gray-700 rounded w-3/4" />
                    <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-full" />
                    <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-5/6" />
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="te-category-archive my-12">
      <div className="container">

        <div className="mb-8">
          <div className="grid grid-cols-[1fr_auto] gap-6 items-end">
            <div>
              {/* Category Header */}
              <SectionHeader
                title={category?.name || slug}
                subtitle={
                  category?.description ||
                  (language === "ar"
                    ? `${POST_TYPE_LABEL} في تصنيف ${category?.name || slug}`
                    : `${POST_TYPE_LABEL} in ${category?.name || slug} category`
                  )
                }
                level="h2"
                marginBottom="0"
                textAlign="start"
              />
            </div>

            <div className="w-auto">
              {/* Category Info & Stats */}
              {category?.count !== undefined && (
                <div className="flex items-center gap-4 text-sm text-gray-600 dark:text-gray-400">
                  <span>
                    {language === "ar"
                      ? `${category.count} عنصر في هذا التصنيف`
                      : `${category.count} Item${category.count !== 1 ? 's' : ''} in this category`
                    }
                  </span>
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Posts Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
          {loading
            ? Array.from({ length: LIMIT }).map((_, i) => (
              <div
                key={i}
                className="animate-pulse bg-white dark:bg-gray-800 rounded-lg shadow-md overflow-hidden"
              >
                <div className="aspect-[3/2] w-full bg-gray-200 dark:bg-gray-700" />
                <div className="p-6 space-y-4">
                  <div className="h-6 bg-gray-200 dark:bg-gray-700 rounded w-3/4" />
                  <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-full" />
                  <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-5/6" />
                </div>
              </div>
            ))
            : posts.length > 0
              ? posts.map((post) => (
                <ContentCard
                  key={post.id}
                  post={post}
                  basePath={POST_TYPE}
                />
              ))
              : (
                <div className="col-span-full">
                  <div className="text-center py-16">
                    <div className="mb-4">
                      <svg
                        className="mx-auto h-12 w-12 text-gray-300 dark:text-gray-600"
                        fill="none"
                        viewBox="0 0 24 24"
                        stroke="currentColor"
                        strokeWidth={1}
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                        />
                      </svg>
                    </div>
                    <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
                      {language === "ar"
                        ? "لا توجد عناصر"
                        : "No items found"
                      }
                    </h3>
                    <p className="text-gray-500 dark:text-gray-400">
                      {language === "ar"
                        ? `لا توجد ${POST_TYPE_LABEL.toLowerCase()} في هذا التصنيف حالياً.`
                        : `No ${POST_TYPE_LABEL.toLowerCase()} found in this category yet.`
                      }
                    </p>
                  </div>
                </div>
              )}
        </div>

        {/* Pagination */}
        {totalPages > 1 && (
          <Pagination
            currentPage={currentPage}
            totalPages={totalPages}
            onPageChange={(page) => setCurrentPage(page)}
          />
        )}
      </div>
    </div>
  );
}